import { ChartIndicator } from "./ChartIndicator";
import { LineSeries } from "../../xy/series/LineSeries";
import * as $array from "../../../core/util/Array";
/**
 * An implementation of a [[StockChart]] indicator.
 *
 * @see {@link https://www.amcharts.com/docs/v5/charts/stock/indicators/} for more info
 */
export class AverageTrueRange extends ChartIndicator {
    constructor() {
        super(...arguments);
        Object.defineProperty(this, "_editableSettings", {
            enumerable: true,
            configurable: true,
            writable: true,
            value: [{
                    key: "period",
                    name: this.root.language.translateAny("Period"),
                    type: "number"
                }, {
                    key: "seriesColor",
                    name: this.root.language.translateAny("Color"),
                    type: "color"
                }]
        });
    }
    _afterNew() {
        this._themeTags.push("averagetruerange");
        super._afterNew();
    }
    _createSeries() {
        return this.panel.series.push(LineSeries.new(this._root, {
            themeTags: ["indicator"],
            xAxis: this.xAxis,
            yAxis: this.yAxis,
            valueXField: "valueX",
            valueYField: "atr",
            fill: undefined
        }));
    }
    _getDataArray(dataItems) {
        const data = [];
        $array.each(dataItems, (dataItem) => {
            data.push({ valueX: dataItem.get("valueX"), value_close: dataItem.get("valueY"), value_high: dataItem.get("highValueY"), value_low: dataItem.get("lowValueY") });
        });
        return data;
    }
    /**
     * @ignore
     */
    prepareData() {
        super.prepareData();
        if (this.series) {
            let period = this.get("period", 20);
            const stockSeries = this.get("stockSeries");
            const dataItems = stockSeries.dataItems;
            let data = this._getDataArray(dataItems);
            let i = 0;
            let index = 0;
            let tr = 0;
            let prevClose;
            let prevATR;
            $array.each(data, (dataItem) => {
                let valueClose = dataItem.value_close;
                if (valueClose != null && prevClose != null) {
                    i++;
                    tr = Math.max(dataItem.value_high - dataItem.value_low, Math.abs(dataItem.value_high - prevClose), Math.abs(dataItem.value_low - prevClose));
                    dataItem.tr = tr;
                    if (i >= period) {
                        if (i == period) {
                            let sum = 0;
                            let k = 0;
                            for (let j = index; j >= 0; j--) {
                                sum += data[j].tr;
                                k++;
                                if (k == period) {
                                    break;
                                }
                            }
                            dataItem.atr = sum / period;
                        }
                        else {
                            if (prevATR != null) {
                                dataItem.atr = (prevATR * (period - 1) + tr) / period;
                            }
                        }
                        prevATR = dataItem.atr;
                    }
                }
                prevClose = valueClose;
                index++;
            });
            this.series.data.setAll(data);
        }
    }
}
Object.defineProperty(AverageTrueRange, "className", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: "AverageTrueRange"
});
Object.defineProperty(AverageTrueRange, "classNames", {
    enumerable: true,
    configurable: true,
    writable: true,
    value: ChartIndicator.classNames.concat([AverageTrueRange.className])
});
//# sourceMappingURL=AverageTrueRange.js.map